// Universal ID Collector - Background Script (ОПТИМИЗИРОВАННЫЙ)
console.log('Universal ID Collector background script loaded (оптимизированный модульный сервер)');

const EXTENSION_SIGNING_SECRET = 'uic_v2_signing_key_8fa3d7c91e';

function stableStringify(value) {
    if (value === null || value === undefined) return '';
    if (typeof value !== 'object') return JSON.stringify(value);
    if (Array.isArray(value)) return `[${value.map(stableStringify).join(',')}]`;

    const keys = Object.keys(value).sort();
    const pairs = keys.map((key) => `${JSON.stringify(key)}:${stableStringify(value[key])}`);
    return `{${pairs.join(',')}}`;
}

async function hmacHex(secret, message) {
    const encoder = new TextEncoder();
    const key = await crypto.subtle.importKey(
        'raw',
        encoder.encode(secret),
        { name: 'HMAC', hash: 'SHA-256' },
        false,
        ['sign']
    );
    const signature = await crypto.subtle.sign('HMAC', key, encoder.encode(message));
    const bytes = new Uint8Array(signature);
    return Array.from(bytes).map((b) => b.toString(16).padStart(2, '0')).join('');
}

async function sha256Hex(message) {
    const encoder = new TextEncoder();
    const digest = await crypto.subtle.digest('SHA-256', encoder.encode(message));
    const bytes = new Uint8Array(digest);
    return Array.from(bytes).map((b) => b.toString(16).padStart(2, '0')).join('');
}

async function buildSignedHeaders({ method, url, body }) {
    const ts = Date.now().toString();
    const nonce = (crypto && crypto.randomUUID)
        ? crypto.randomUUID().replace(/-/g, '')
        : `${Math.random().toString(36).slice(2)}${Date.now().toString(36)}`;
    const bodyRaw = stableStringify(body);
    const bodyHash = await sha256Hex(bodyRaw);
    let pathAndQuery = url;
    try {
        const parsed = new URL(url);
        pathAndQuery = `${parsed.pathname}${parsed.search || ''}`;
    } catch (e) {
        pathAndQuery = url;
    }
    const payload = `${String(method || 'GET').toUpperCase()}\n${pathAndQuery}\n${ts}\n${nonce}\n${bodyHash}`;
    const signature = await hmacHex(EXTENSION_SIGNING_SECRET, payload);

    return {
        'X-Ext-Ts': ts,
        'X-Ext-Nonce': nonce,
        'X-Ext-Signature': signature
    };
}

// Единый обработчик сообщений
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    console.log('Background received message:', request.action);
    
    switch (request.action) {
        case 'extensionInitialized':
            console.log(`Расширение инициализировано на ${request.site}`);
            console.log('API статус:', request.apiStatus);
            break;
            
        case 'parsersUpdated':
        case 'settingsUpdated':
            // Пересылаем в popup если открыт
            chrome.runtime.sendMessage(request).catch(error => {
                if (!error.message.includes('Receiving end does not exist')) {
                    console.warn('Error forwarding to popup:', error);
                }
            });
            break;
            
        case 'showNotification':
        case 'toggleAutoScan':
        case 'rescanPage':
            // Пересылаем в активную вкладку
            chrome.tabs.query({ active: true, currentWindow: true }).then(tabs => {
                if (tabs[0]?.id) {
                    chrome.tabs.sendMessage(tabs[0].id, request).catch(error => {
                        if (!error.message.includes('Receiving end does not exist')) {
                            console.warn('Error forwarding to tab:', error);
                        }
                    });
                }
            });
            break;
            
        case 'checkServer':
            // Пересылаем в active tab для проверки
            chrome.tabs.query({ active: true, currentWindow: true }).then(tabs => {
                if (tabs[0]?.id) {
                    chrome.tabs.sendMessage(tabs[0].id, request, (response) => {
                        if (response) {
                            sendResponse(response);
                        }
                    });
                }
            });
            return true; // Важно для асинхронного ответа
        case 'apiRequest':
            (async () => {
                try {
                    const store = await new Promise(r => chrome.storage.local.get(['apiKey','serverUrl','clientId'], r));
                    const apiKey = store.apiKey || '';
                    // Если ключ отсутствует — не пробуем обращаться к серверу, возвращаем 401
                    if (!apiKey || String(apiKey).trim() === '') {
                        sendResponse({ status: 401, error: 'missing_api_key' });
                        return;
                    }
                    let clientId = store.clientId || '';
                    if (!clientId) {
                        clientId = (crypto && crypto.randomUUID) ? crypto.randomUUID() : `cid_${Math.random().toString(36).slice(2)}${Date.now().toString(36)}`;
                        chrome.storage.local.set({ clientId });
                    }
                    const serverUrl = store.serverUrl || 'https://idmanager.site';
                    const url = request.url && request.url.startsWith('http') ? request.url : (serverUrl + (request.url || '/'));
                    const method = String(request.method || 'GET').toUpperCase();
                    const body = request.body || null;
                    const signedHeaders = await buildSignedHeaders({ method, url, body });
                    const res = await fetch(url, {
                        method,
                        headers: Object.assign({'Content-Type':'application/json','X-API-Key': apiKey, 'X-Client-Id': clientId}, signedHeaders, request.headers || {}),
                        body: body ? JSON.stringify(body) : undefined,
                        credentials: 'omit'
                    });
                    const text = await res.text();
                    let data;
                    try { data = JSON.parse(text); } catch(e){ data = text; }
                    if (res.status === 401 || res.status === 402 || (data && data.error === 'subscription_expired')) {
                        chrome.runtime.sendMessage({ action: 'subscriptionExpired', details: data });
                    }
                    sendResponse({ status: res.status, data });
                } catch (err) {
                    sendResponse({ status: 0, error: String(err) });
                }
            })();
            return true;
    }
    
    // Не держим соединение открытым для синхронных сообщений
    if (request.action !== 'checkServer') {
        return false;
    }
});

// Обработчик установки
chrome.runtime.onInstalled.addListener((details) => {
    console.log('Extension installed/updated:', details.reason);
    
    if (details.reason === 'install') {
        // Устанавливаем настройки по умолчанию
        chrome.storage.local.set({
            enabled: true,
            firstInstall: true
        });
    }
});

console.log('Background script ready');
